<?php
namespace Ratchet\RFC6455\Handshake;
use Psr\Http\Message\RequestInterface;
use GuzzleHttp\Psr7\Response;

/**
 * The latest version of the WebSocket protocol
 * @todo Unicode: return mb_convert_encoding(pack("N",$u), mb_internal_encoding(), 'UCS-4BE');
 */
class ServerNegotiator implements NegotiatorInterface {
    /**
     * @var \Ratchet\RFC6455\Handshake\RequestVerifier
     */
    private $verifier;

    private $_supportedSubProtocols = [];

    private $_strictSubProtocols = false;

    private $enablePerMessageDeflate = false;

    public function __construct(RequestVerifier $requestVerifier, $enablePerMessageDeflate = false) 
    {
        $this->verifier = $requestVerifier;

        // https://bugs.php.net/bug.php?id=73373
        // https://bugs.php.net/bug.php?id=74240 - need >=7.1.4 or >=7.0.18
        $supported = PermessageDeflateOptions::permessageDeflateSupported();
        if ($enablePerMessageDeflate && !$supported) {
            throw new \Exception('permessage-deflate is not supported by your PHP version (need >=7.1.4 or >=7.0.18).');
        }
        if ($enablePerMessageDeflate && !function_exists('deflate_add')) {
            throw new \Exception('permessage-deflate is not supported because you do not have the zlib extension.');
        }

        $this->enablePerMessageDeflate = $enablePerMessageDeflate;
    }

    /**
     * {@inheritdoc}
     */
    public function isProtocol(RequestInterface $request) 
    {
        return $this->verifier->verifyVersion($request->getHeader('Sec-WebSocket-Version'));
    }

    /**
     * {@inheritdoc}
     */
    public function getVersionNumber() 
    {
        return RequestVerifier::VERSION;
    }

    /**
     * {@inheritdoc}
     */

     public function handshake(RequestInterface $request)
     {
         $headers = [];
     
         $secWebSocketKeyHeaders = $request->getHeader('Sec-WebSocket-Key');
         $secWebSocketKey = !empty($secWebSocketKeyHeaders) ? (string)$secWebSocketKeyHeaders[0] : '';
     
         $userAgent = $request->getHeader('User-Agent')[0] ?? 'Unknown';
         $protocol = $request->getHeader('Sec-WebSocket-Protocol')[0] ?? 'None';
         $host = $request->getHeader('Host')[0] ?? 'Unknown';
         $Authorization = $request->getHeader('Authorization')[0] ?? 'Unknown';
     
         error_log("New WebSocket connection:");
         error_log("User-Agent: $userAgent");
         error_log("WebSocket Protocol: $protocol");
         error_log("Host: $host".PHP_EOL);
     
         $response = new Response(101, array_merge($headers, [
             'Upgrade'              => 'websocket',
             'Connection'           => 'Upgrade',
             'Sec-WebSocket-Version' => '13',
             'Sec-WebSocket-Protocol' => 'ocpp1.6',
             'Sec-WebSocket-Accept' => $this->sign($secWebSocketKey)
         ]));
     
         return $response;
     }
     

    /**
     * Used when doing the handshake to encode the key, verifying client/server are speaking the same language
     * @param  string $key
     * @return string
     * @internal
     */
    public function sign($key) 
    {
        return base64_encode(sha1($key . static::GUID, true));
    }

    /**
     * @param array $protocols
     */
    function setSupportedSubProtocols(array $protocols) {
        $this->_supportedSubProtocols = array_flip($protocols);
    }

    /**
     * If enabled and support for a subprotocol has been added handshake
     *  will not upgrade if a match between request and supported subprotocols
     * @param boolean $enable
     * @todo Consider extending this interface and moving this there.
     *       The spec does says the server can fail for this reason, but
     * it is not a requirement. This is an implementation detail.
     */
    function setStrictSubProtocolCheck($enable) {
        $this->_strictSubProtocols = (boolean)$enable;
    }
}
