<?php
namespace MyApp;

class Subscription
{
    private $db;

    public function __construct($db)
    {
        $this->db = $db;
    }

    /**
     * Handle charging based on user's subscription if available
     *
     * @param int $user_id User ID
     * @param string $idTag Charger ID tag
     * @param int $transactionId Transaction ID
     * @param float $energy Energy used in this increment (in kWh or Wh based on unit)
     * @param string $unit Energy unit (Wh or kWh)
     * @param float $price_per_unit Price per unit of energy
     * @return array Result with pricing details and subscription usage
     */
    public function handleChargingWithSubscription($user_id, $idTag, $transactionId, $energy, $unit, $price_per_unit)
    {
        $result = [
            'subscription_used' => false,
            'subscription_id' => null,
            'subscription_kwh_used' => 0,
            'balance_charge' => 0,
            'energy_unit' => $unit,
            'payment_type' => 'wallet' // Default to wallet payment
        ];

        // Skip if no energy used
        if ($energy <= 0) {
            return $result;
        }

        // Get charger company ID for subscription matching
        $charger = $this->db->row("SELECT company_id FROM `my_charger` WHERE idTag = '{$idTag}'");
        $charger_company_id = $charger['company_id'] ?? 0;

        if ($charger_company_id <= 0) {
            // No company ID, charge directly from wallet
            $result['balance_charge'] = $energy * $price_per_unit;
            return $result;
        }

        // Convert energy to kWh for subscription calculation if needed
        $energy_kwh = $energy;
        if ($unit === 'Wh') {
            $energy_kwh = $energy / 1000;
        }

        // Find active subscription for this user with the same company as the charger
        $subscription = $this->db->row("
            SELECT su.*, s.company_id, s.kwh_allowance, s.name
            FROM subscriptions_users su
            JOIN subscriptions s ON su.subscription_id = s.id
            WHERE su.user_id = '{$user_id}'
            AND su.is_active = 1
            AND su.remaining_kwh > 0
            AND s.company_id = '{$charger_company_id}'
            AND su.expiration_date >= NOW()
            ORDER BY su.id DESC
            LIMIT 1
        ");

        if (!$subscription) {
            // No matching subscription, charge directly from wallet
            $result['balance_charge'] = $energy * $price_per_unit;
            return $result;
        }

        // We found a subscription
        $subscription_id = $subscription['subscription_id'];
        $user_subscription_id = $subscription['id'];
        $available_kwh = $subscription['remaining_kwh'];

        if ($available_kwh >= $energy_kwh) {
            // Full amount can be covered by subscription
            $new_remaining = $available_kwh - $energy_kwh;

            // Update subscription remaining kWh
            $this->db->query("
                UPDATE subscriptions_users
                SET remaining_kwh = '{$new_remaining}',
                    updated_at = NOW()
                WHERE id = '{$user_subscription_id}'
            ");

            $result['subscription_used'] = true;
            $result['subscription_id'] = $subscription_id;
            $result['subscription_kwh_used'] = $energy_kwh;
            $result['payment_type'] = 'subscription';
        } else if ($available_kwh > 0) {
            // Partial amount can be covered by subscription
            $subscription_kwh = $available_kwh;

            // Calculate remaining amount to charge from balance
            $remaining_energy = $energy_kwh - $available_kwh;
            $remaining_in_original_unit = $remaining_energy;
            if ($unit === 'Wh') {
                $remaining_in_original_unit = $remaining_energy * 1000;
            }

            // Update price to charge from balance
            $balance_charge = $remaining_in_original_unit * $price_per_unit;

            // Update subscription - set to 0 as we've used all available kWh
            $this->db->query("
                UPDATE subscriptions_users
                SET remaining_kwh = 0,
                    updated_at = NOW()
                WHERE id = '{$user_subscription_id}'
            ");

            $result['subscription_used'] = true;
            $result['subscription_id'] = $subscription_id;
            $result['subscription_kwh_used'] = $subscription_kwh;
            $result['balance_charge'] = $balance_charge;
            $result['payment_type'] = 'mixed';
        }

        return $result;
    }
}
